/****************************************************************************
*****************************************************************************
**
** File Name
** ---------
**
** LinkList.h
**
*****************************************************************************
*****************************************************************************
**
** Description
** -----------
**
** Linked List public interface definitions.
** Both single and double linked lists are supported.
**
** Linked lists are lists implemented by stringing the list linkage
** through the first few bytes of each entry.  Both single and double linked
** lists are supported.
**
** The single linked list structures are appropriate for any queuing
** application where the FIFO order of the list will not be violated.  In
** this case, deletions are never made from the middle of the list. The head
** pointer for this type of queue contains TWO pointers: one to the head
** element and another to the tail element.  All other elements contain only
** one pointer: to the successor ("next").
**
** The double linked list structures are appropriate for applications that
** will make deletions at random points in the list, without having to
** search the list.  This can be accomplished by keeping two pointers in
** each list element: a successor pointer ("next") and a predecessor
** pointer ("prev").
**
** In handling linked lists, it is important to be able to delete ANY list
** element without having to deal with deletion from the head as a special
** case.  This is accomplished by reserving the first element of each list
** (pointed to by "head") as a "dummy" element.  This element is essentially
** just a list head pointer.  The condition for an empty queue is:
**
**      next( head ) = pointer to head itself
**
** If a given list is empty, deletion from it becomes a no-op.  These linked
** list implementations are universal and can be used to create lists whose
** elements are all different types and different sizes.  Of course, this
** implementation applies equally well to lists of uniform type.  The first
** few bytes of each list element must be dedicated to the list pointers.
**
** Typedefs for linked list elements should include as their first structure
** component the type "LL_SINGLE_LINK_HEADER" or "LL_DOUBLE_LINK_HEADER"
** defined below.
**
** All of the simpler list services are implemented as macros (init, empty,
** not_empty, etc.), but the basic put, take, and remove routines are
** implemented as functions.
**
*****************************************************************************
*****************************************************************************
**
** Source Change Indices
** ---------------------
**
** Porting: <none>0----<major>         Customization: <none>0----<major>
**
*****************************************************************************
*****************************************************************************
**
** Services List
** -------------
**
** Single or double linked list services:
**
**    LL_Empty()                    Check if a linked list is empty
**    LL_EndOfList()                Check if end of list has been reached
**    LL_FirstEntry()               Get first entry in linked list
**    LL_Init()                     Initialize a linked list
**    LL_NextEntry()                Walk to the next entry in linked list
**
** Double linked list services:
**
**    LL_PutDoubleLinkList()        Put a new entry at tail of list
**    LL_RemoveDoubleLinkList()     Remove any entry from double linked list
**
** Single linked list services:
**
**    LL_PutSingleLinkList()        Put a new entry at tail of list
**    LL_TakeSingleLinkList()       Take entry from head of single linked list
**
*****************************************************************************
*/

#ifndef LINKLIST_H
#define LINKLIST_H



/****************************************************************************
**
** Typedefs
**
*****************************************************************************
*/

/*---------------------------------------------------------------------------
**
** LL_HEAD_NODE
** LL_HEAD_NODE_HEADER
**
** Head node for both singly and doubly linked lists.
**
** A head node may be defined in one of two ways:
** 1) A minimum head node that contains just the head and tail pointers.
**    This is done using the LL_HEAD_NODE macro.
** 2) A head node that is a copy of the structures in the linked list.
**    This is done by including the LL_HEAD_NODE_HEADER in the
**    definition of the head node structure.
**
** Usage examples:
**    1) LL_HEAD_NODE( LinkedFooType ) sFooLinkedList;
**    2) typedef struct LinkedFooType
**       {
**          LL_HEAD_NODE_HEADER( LinkedFooType );
**          ...
**          User defined portion of structure, such as
**          void *pFoo;
**          ...
**       }
**       LinkedFooType;
**
** The next pointer points to the first entry in the list.
** The prev pointer points to the tail entry in the list.
**
**---------------------------------------------------------------------------
*/

#define LL_HEAD_NODE( UserType ) \
struct                           \
{                                \
   struct UserType *pNext;       \
   struct UserType *pPrev;       \
}

#define LL_HEAD_NODE_HEADER( UserType ) \
   struct UserType *pNext;              \
   struct UserType *pPrev



/*---------------------------------------------------------------------------
**
** LL_SINGLE_LINK_HEADER
** LL_DOUBLE_LINK_HEADER
**
** The common portion of any user defined structure that is to be
** managed in singly or doubly linked lists.  This macro is to be included
** as the first part of any linked list user structure.
**
** Usage example:
**    typedef struct LinkedFooType
**    {
**       LL_DOUBLE_LINK_HEADER( LinkedFooType );
**       ...
**       User defined portion of structure, such as
**       void *pFoo;
**       ...
**    }
**    LinkedFooType;
**
**---------------------------------------------------------------------------
*/

#define LL_SINGLE_LINK_HEADER( UserType ) \
   struct UserType *pNext

#define LL_DOUBLE_LINK_HEADER( UserType ) \
   struct UserType *pNext;                \
   struct UserType *pPrev



/*---------------------------------------------------------------------------
**
** LL_HEAD_NODE_SIZE
** LL_SINGLE_LINK_HEADER_SIZE
** LL_DOUBLE_LINK_HEADER_SIZE
**
** Sizes of the various portions of the linked list management overhead.
**
**---------------------------------------------------------------------------
*/

#define LL_HEAD_NODE_SIZE           ( sizeof( void* ) * 2 )
#define LL_SINGLE_LINK_HEADER_SIZE  ( sizeof( void* ) )
#define LL_DOUBLE_LINK_HEADER_SIZE  ( sizeof( void* ) * 2 )


/*---------------------------------------------------------------------------
**
** ll_DoubleLinkEntryType
**
** Header portion of any double link list entry.
** All user defined double linked list entries contain this
** at their beginning.
**
**---------------------------------------------------------------------------
*/

typedef struct ll_DoubleLinkEntryType
{
   LL_DOUBLE_LINK_HEADER( ll_DoubleLinkEntryType );
}
ll_DoubleLinkEntryType;



/*---------------------------------------------------------------------------
**
** ll_HeadNodeType
**
** Initial portion of any head node.
**
**---------------------------------------------------------------------------
*/

typedef struct ll_HeadNodeType
{
   LL_HEAD_NODE_HEADER( ll_HeadNodeType );
}
ll_HeadNodeType;



/*---------------------------------------------------------------------------
**
** ll_SingleLinkEntryType
**
** Header portion of any single link list entry.
** All user defined single linked list entries contain this
** at their beginning.
**
**---------------------------------------------------------------------------
*/

typedef struct ll_SingleLinkEntryType
{
   LL_SINGLE_LINK_HEADER( ll_SingleLinkEntryType );
}
ll_SingleLinkEntryType;


/****************************************************************************
**
** Services
**
*****************************************************************************
*/

#ifdef __cplusplus
#define EXTFUNC extern "C"
#else
#define EXTFUNC extern
#endif

/*---------------------------------------------------------------------------
**
** LL_Empty()
**
** This service tests if a double or single linked list is empty.
**
**---------------------------------------------------------------------------
**
** Inputs:
**    pH                      - pointer to dummy head node of the list
**
** Outputs:
**    Return                  - TRUE  if the list is empty
**                              FALSE if the list contains at least 1 entry
**
** Usage:
**    if( LL_Empty( pH ) ){ ... }
**
**---------------------------------------------------------------------------
*/

#define LL_Empty( pH )  \
   ( (pH)->pNext == (void*)(pH) )



/*---------------------------------------------------------------------------
**
** LL_EndOfList()
**
** Determine if we've reached the end of the linked list.
**
**---------------------------------------------------------------------------
**
** Inputs:
**    pHead                   - Pointer to list's dummy head node
**    pEntry                  - Pointer to entry in list
**
** Outputs:
**    Return                  - TRUE  if end of list reached
**                                    (we're back to the dummy head node)
**                              FALSE if there are more entries in the list
**
** Usage:
**    while( !LL_EndOfList( pHead, pEntry ) )
**
**---------------------------------------------------------------------------
*/

#define LL_EndOfList( pHead, pEntry )  \
   ( (void*)(pEntry) == (void*)(pHead) )



/*---------------------------------------------------------------------------
**
** LL_FirstEntry()
**
** Get the first entry in a linked list.
** If the list is empty, a pointer to the dummy head node is returned.
** This service should be preceeded by a call to LL_NotEmpty or followed
** by a call to LL_NotLastEntry to check for this condition.
**
**---------------------------------------------------------------------------
**
** Inputs:
**    pH                      - Pointer to dummy head node of the list
**
** Outputs:
**    Return                  - Pointer to first entry in linked list
**                              or pointer to the head node if list empty
**
** Usage:
**    pEntry = LL_FirstEntry( pHead )
**
**---------------------------------------------------------------------------
*/

#define LL_FirstEntry( pHead )  \
   ( (pHead)->pNext )


/*---------------------------------------------------------------------------
**
** LL_LastEntry()
**
** Get the last entry in a linked list.
** If the list is empty, a pointer to the dummy head node is returned.
** This service should be preceeded by a call to LL_NotEmpty or followed
** by a call to LL_NotLastEntry to check for this condition.
**
**---------------------------------------------------------------------------
**
** Inputs:
**    pH                      - Pointer to dummy head node of the list
**
** Outputs:
**    Return                  - Pointer to last entry in linked list
**                              or pointer to the head node if list empty
**
** Usage:
**    pEntry = LL_LastEntry( pHead )
**
**---------------------------------------------------------------------------
*/

#define LL_LastEntry( pHead )  \
   ( (pHead)->pPrev )




/*---------------------------------------------------------------------------
**
** LL_Init()
**
** This service initializes a double or single linked list to its default
** empty state.  An empty list has the head and tail pointers of the
** dummy head node pointing to the dummy head node.
**
**---------------------------------------------------------------------------
**
** Inputs:
**    pH                      - pointer to dummy head node of the list
**
** Outputs:
**    None
**
** Usage:
**    LL_Init( pHead )
**
**---------------------------------------------------------------------------
*/

#define LL_Init( pHead )            \
{                                   \
   (pHead)->pNext = (void*)(pHead); \
   (pHead)->pPrev = (void*)(pHead); \
}



/*---------------------------------------------------------------------------
**
** LL_NextEntry()
**
** Walk a linked list one entry ahead.
**
**---------------------------------------------------------------------------
**
** Inputs:
**    pEntry                  - pointer to entry in list
**
** Outputs:
**    None
**
** Usage:
**    pEntry = LL_NextEntry( pEntry )
**
**---------------------------------------------------------------------------
*/

#define LL_NextEntry( pEntry )  \
   ( (pEntry)->pNext )



/*---------------------------------------------------------------------------
**
** LL_PutDoubleLinkList()
** LL_PutSingleLinkList()
**
** This service inserts a new element into a double or single linked list
** at the tail end of the list.  The list is of a circular type
** with a dummy head node that avoids boundary conditions for empty lists.
**
**---------------------------------------------------------------------------
**
** Inputs:
**    pHead                   - pointer to the dummy head node of the list
**    pEntry                  - pointer to the new entry to be added
**
** Outputs:
**    None
**
** Usage:
**    LL_PutDoubleLinkList( pHead, pEntry )
**    LL_PutSingleLinkList( pHead, pEntry )
**
**---------------------------------------------------------------------------
*/

EXTFUNC void LL_PutDoubleLinkList( void *pHead, void *pEntry );
EXTFUNC void LL_PutSingleLinkList( void *pHead, void *pEntry );



/*---------------------------------------------------------------------------
**
** LL_RemoveDoubleLinkList()
**
** This service removes an arbitrary entry anywhere in a double linked
** list.  The list is of a circular type with a dummy head
** node that avoids boundary conditions for empty lists.  No error checking
** on the input element pointer is performed and unpredictable execution will
** follow invocation of this service with an invalid pointer.  The input
** element pointer MUST point to a double linked list element with the
** first portion of the type LL_DOUBLE_LINK_HEADER.
**
**---------------------------------------------------------------------------
**
** Inputs:
**    pX                      - pointer to entry to remove
**
** Outputs:
**    None
**
** Usage:
**    LL_RemoveDoubleLinkList( pEntry )
**
**---------------------------------------------------------------------------
*/

EXTFUNC void LL_RemoveDoubleLinkList( void *pEntry );



/*---------------------------------------------------------------------------
**
** LL_TakeSingleLinkList()
**
** This service removes the next element at the head of a single linked
** list.  The list is of a circular type with a dummy head node that avoids
** boundary conditions for empty lists.  A pointer to the removed entry
** is returned.
**
**---------------------------------------------------------------------------
**
** Inputs:
**    pHead                   - pointer to the dummy head node of the list
**
** Outputs:
**    Return                  - pointer to the removed element
**
** Usage:
**    pEntry = LL_TakeSingleLinkList( pHead );
**
**---------------------------------------------------------------------------
*/

EXTFUNC void *LL_TakeSingleLinkList( void *pHead );



#endif  /* inclusion lock */
