/*
|============================================================================
|
|       Copyright (C) 2011 ProSoft Technology. All rights reserved.
|
|  File:             ntpset.c
|
|  Class(es):        
|
|  Inherits From:    
|
|  Summary:          
|
|  Project(s):       Common Utility
|
|  Subsystem:        ProLinx
|
|  Contributors:     Henry Yu(HYU)
|
|  Description:      This file contains the definitions needed for NTP.
|
|  Notes:            
|
|
|============================================================================
|  Version     Date     Author  Change    Description
|----------------------------------------------------------------------------
|  Build     12/06/2011 HYU      Created.
|============================================================================
*/

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "plxdef.h"
#include "ntpset.h"
#include "cfgparse.h"
#include "plxutil.h"



#define NTP_STAT_FILE_NAME "/etc/ntp/peerstats"

#define 	NTP_UPDATE_MAX		600*1000000L  	// 600 seconds
#define		NTP_MAX_OFFSET		60				// sixty (60) seconds


static NTP_CONFIG_DATA my_ntp_data;
static NTP_STATUS_DATA ntp_status_data;


//Get data from the [SNTP CLIENT] section of the configuration file.
int NTP_LoadConfig( char * fname, NTP_CONFIG_DATA * config, int Max_DB_addr )
{
    int ret = 0;
    
    memset(&my_ntp_data, 0 , sizeof(my_ntp_data));
    //   strcpy(my_ntp_data.ntp_ip, "132.163.4.102"); //NIST, Boulder, Colorado
    strcpy(my_ntp_data.ntp_ip, "0.0.0.0");
    my_ntp_data.status_offset = -1;

    ConfigParser cp;
    char *cptr;
    char c;
    
    ConfigParser_Init(&cp, ':');
    
    ret = ConfigParser_LoadSection( &cp, fname, "[SNTP CLIENT]");
    if (ret)
    {
        cptr = ConfigParser_SeekItemGetValue( &cp, "NTP SERVER IP ADDRESS");
        if(cptr)  //item is found
        {
            strcpy(my_ntp_data.ntp_ip, cptr);
        }
        else
        {
            strcpy(my_ntp_data.ntp_ip, "0.0.0.0");
        }
        
        ConfigParser_SeekItemGetShort(&cp, "TIME ZONE", &my_ntp_data.ntp_timeZone);

        if(my_ntp_data.ntp_timeZone > 11 || my_ntp_data.ntp_timeZone < -11)
            my_ntp_data.ntp_timeZone = 0;
        
        if (ConfigParser_SeekItemGetChar(&cp, "USE DAYLIGHT SAVINGS TIME", &c) && c == 'Y')
        {
            my_ntp_data.ntp_dst = 1;
        }

        if ( Max_DB_addr > 0 && ConfigParser_SeekItemGetInt(&cp, "DATABASE REGISTER", &my_ntp_data.status_offset))
        {
            if (( my_ntp_data.status_offset < 0 ) || ( my_ntp_data.status_offset >= Max_DB_addr ))
                my_ntp_data.status_offset = -1;
        }
        else
        {
            my_ntp_data.status_offset = -1;
        }

        if (config != NULL)
        {
            *config = my_ntp_data;
        }
    }

    ConfigParser_Cleanup(&cp);
    
    return ret;
}

int NTP_ApplyConfig(char * fname)
{
    char tzone_str[20];
    
    NTP_LoadConfig(fname, NULL, -1);
    
#ifdef __linux__
    if(my_ntp_data.ntp_dst)
        sprintf(tzone_str,"TZN%dDST", my_ntp_data.ntp_timeZone);
    else
        sprintf(tzone_str,"TZN%d", my_ntp_data.ntp_timeZone);
    
    setenv("TZ", tzone_str, 1);
#else
    if(my_ntp_data.ntp_dst)
        sprintf(tzone_str,"TZ=TZN%dDST", my_ntp_data.ntp_timeZone);
    else
        sprintf(tzone_str,"TZ=TZN%d", my_ntp_data.ntp_timeZone);
    
    putenv(tzone_str);
#endif

    tzset();

    return 0;
}


int ediag_ntp_config(char * buff, int buff_len)
{
    int count, len = buff_len;
    
    count = snprintf(buff, len, "\nSNTP CLIENT CONFIGURATION:\n"
        "  NTP SERVER IP : %s\n"
        "  TIME ZONE     : %-3d      USE DST       : %-3s\n",
        my_ntp_data.ntp_ip,
        my_ntp_data.ntp_timeZone,
        my_ntp_data.ntp_dst ? "Yes" : "No");
    
    buff += count; len -= count;
    
    time_t  t;
    time(&t);
    
    struct tm *area;
    area = localtime((time_t *)&t);
    
    count = snprintf(buff, len, "SNTP Local time is: %s\n", asctime(area));
    buff += count; len -= count;
    
    area = gmtime((time_t *)&t);
    
    count = snprintf(buff, len, "SNTP GMT is:        %s\n", asctime(area));
    buff += count; len -= count;
    
    return buff_len - len;
}

int diagNtpConfig(char * buff, int buff_len)
{
    int count, len = buff_len;
    
    count = snprintf(buff, len, "NTP SERVER IP,%s,"
        "TIME ZONE,%-3d,USE DST,%-3s,",
        my_ntp_data.ntp_ip,
        my_ntp_data.ntp_timeZone,
        my_ntp_data.ntp_dst ? "Yes" : "No");
    
    buff += count; len -= count;
    
    time_t  t;
    time(&t);
    
    char timeBuff[32] = { 0 };
    struct tm *area;
    area = localtime((time_t *)&t);
    
    strncpy(timeBuff, asctime(area), 31);
    timeBuff[31] = '\0';
    TrimRight(timeBuff);

    count = snprintf(buff, len, "SNTP Local Time,%s,", timeBuff);
    buff += count; len -= count;
    
    area = gmtime((time_t *)&t);
    
    strncpy(timeBuff, asctime(area), 31);
    timeBuff[31] = '\0';
    TrimRight(timeBuff);

    count = snprintf(buff, len, "SNTP GMT,%s,", timeBuff);
    buff += count; len -= count;
    
    return buff_len - len;
}


int ediag_ntp_status(char * buff, int buff_len)
{
    int count, len = buff_len;

    ntp_update_status();
    
    count = snprintf( buff, len, "\nSNTP CLIENT STATUS:\n"
        "  NTP SERVER IP : %s\n"
        "  TIME ZONE     : %-3d      USE DST       : %-3s\n"
        "  NTP Valid     : %s\n"
        "  Timeout Err Count: %d\n"
        "  Seconds offset from NTP Time: %lld\n",
        my_ntp_data.ntp_ip,
        my_ntp_data.ntp_timeZone,
        my_ntp_data.ntp_dst ? "Yes" : "No",
        (ntp_status_data.valid ? "Yes" : "No"),
        ntp_status_data.tmout_count,
        ntp_status_data.offset );
    
    buff += count; len -= count;

    time_t  t;
    time(&t);
    
    struct tm *area;
    area = localtime((time_t *)&t);
    
    count = snprintf(buff, len, "SNTP Local time is: %s\n", asctime(area));
    buff += count; len -= count;
    
    area = gmtime((time_t *)&t);
    
    count = snprintf(buff, len, "SNTP GMT is:        %s\n", asctime(area));
    buff += count; len -= count;

    return buff_len - len;
}


int diagNtpStatus(char * buff, int buff_len)
{
    int count, len = buff_len;

    ntp_update_status();

    count = snprintf( buff, len,
        "NTP SERVER IP,%s,"
        "TIME ZONE,%d,USE DST,%s,"
        "NTP Valid,%s,"
        "Timeout Err Count,%d,"
        "Seconds offset from NTP Time,%lld,",
        my_ntp_data.ntp_ip,
        my_ntp_data.ntp_timeZone,
        my_ntp_data.ntp_dst ? "Yes" : "No",
        (ntp_status_data.valid ? "Yes" : "No"),
        ntp_status_data.tmout_count,
        ntp_status_data.offset );
    
    buff += count; len -= count;

    time_t  t;
    time(&t);
    
    char timeBuff[32];
    struct tm *area;
    area = localtime((time_t *)&t);
    
    strncpy(timeBuff, asctime(area), 31);
    timeBuff[31] = '\0';
    TrimRight(timeBuff);
    
    count = snprintf(buff, len, "SNTP Local Time,%s,", timeBuff);
    buff += count; len -= count;
    
    area = gmtime((time_t *)&t);
    
    strncpy(timeBuff, asctime(area), 31);
    timeBuff[31] = '\0';
    TrimRight(timeBuff);
    
    count = snprintf(buff, len, "SNTP GMT,%s,", timeBuff);
    buff += count; len -= count;
    
    return buff_len - len;
}


int ntp_update_status( void )
{
    FILE * 				fptr;
    long 				FLoc			= 0;
    char 				buff[256];
    int					result 			= -1;
    static long long	last_update 	= 0;
    static long			prev_seconds 	= -1;
    static long			prev_day 		= -1;
    long				day;
    long				seconds;
    long long			offset 			= NTP_MAX_OFFSET;
    char *				cptr;
    
    
    // read last rawstats update day and time(seconds)
    // from the file updated by the ntpdaemon
    
    fptr = fopen( NTP_STAT_FILE_NAME, "rt" );
    if ( fptr != NULL )
    {
        // parse string of format
        // modified_julian_day mod_julian_UTC_time_sec  IP_of_peer status  time_offset_secs  delay  dispersion rms_jitter
        // which looks like
        // 48773 10847.650 127.127.4.1 9714 -0.001605 0.00000 0.00142
        // we care about
        //   1) the day and sec: which gives the time/date of the last update
        //   2) the time_offset: which tells us how far off the clock and peer are.
        
        while( fgets( buff, 255, fptr ))
        {
            FLoc = ftell(fptr);
        }
        fseek( fptr, FLoc, SEEK_SET );
        fgets( buff, 255, fptr );
        
        fclose(fptr);
        
        // read day and seconds
        day = atoi( buff );
        cptr = strchr( buff, ' ');
        if ( ! cptr )
            return( result );
        
        seconds = atol( cptr );
        
        cptr = strchr( cptr+1, ' ' );	// skip fractional part of seconds
        if ( ! cptr )
            return( result );
        
        cptr = strchr( cptr+1, ' ' );	// skip ip address
        if ( ! cptr )
            return( result );
        
        cptr = strchr( cptr+1, ' ' );	// skip status
        if ( ! cptr )
            return( result );
        
        // read offset
        offset = atoll( cptr );
        ntp_status_data.offset = offset;

        if ( last_update == 0 )
        {
            // first update
            getElapsedTimeFrom( &last_update );
            prev_day 		= day;
            prev_seconds 	= seconds;
            
            if ( llabs( offset ) < NTP_MAX_OFFSET )
                ntp_status_data.valid = 1;
            else
                ntp_status_data.valid = 0;
            
            result 			= 0;
        }
        else
        {
            if (( day != prev_day ) || (seconds != prev_seconds ))
            {
                // new update to stats file, so update status/time of sample.
                
                prev_day 		= day;
                prev_seconds 	= seconds;
                
                getElapsedTimeFrom( &last_update );  // get the current time
                
                // update status data
                // if server and our clock are off by 60 seconds or more, then time is invalid,
                //  otherwise set our time as valid
                if (llabs( offset ) < NTP_MAX_OFFSET )
                {
                    if ( ntp_status_data.valid == 0 )
                    {
                        printf("Changing ntp time stat to valid: offset == %lld\n", llabs(offset) );
                    }

                    ntp_status_data.valid = 1;
                }
                else if ( ntp_status_data.valid )
                {
                    ntp_status_data.tmout_count++;
                    ntp_status_data.valid = 0;
                    printf("Changing ntp time stat to INvalid: offset == %lld\n", llabs(offset) );
                }
            }
            else
            {
                if ( getElapsedTimeFrom( &last_update ) > NTP_UPDATE_MAX )
                {
                    // no update has occurred for a long time.  flag event.
                    // increment timeout error
                    ntp_status_data.tmout_count++;
                    ntp_status_data.valid = 0;
                }
            }
            
            result = 0;
        }
    }
    
    return( result );
}

void ntp_get_status( NTP_STATUS_DATA * ntp_status_ptr )
{
    memcpy( ntp_status_ptr, &ntp_status_data, sizeof( NTP_STATUS_DATA ));
}

