/*****************************************************************************
*   Copyright (c) 2011-2012 Prosoft Technology, Inc.
*
*   Title:        Communications Module Serial Port Definitions
*
*   Abstract:     This file contains definitions and declarations
*                 needed for serial port communication.
*
*   Environment:  Linux
*                 GCC C/C++ Compiler
*
*   Author:       HYU
*
*  08/20/2011 HYU - Migrated to ProLinx Linux platform.
*  08/26/2011 HYU - Added spRegisterCallback() for status monitoring.
*  01/05/2012 HYU - Added spSetPortMode() and modified spSetRTS() to change mode for RS485.
*  01/06/2012 HYU - Modified spGetCountUnsent() to check LSR(Line Status Register) to confirm transmit buffer is empty.
*  02/08/2012 HYU - Added spGetDMACount().
*  03/28/2012 HYU - Swapped USART1_ADDRESS and USART2_ADDRESS because S1 and S2 ports swapped.
*  04/02/2012 HYU - Changed spSetPortMode() to swap GPIOs for COM1 and COM2.
*  04/03/2012 HYU - Added the definition PORT_1_2_SWAPPED.
*****************************************************************************/

/////////////////////////////////////////////////////////////////////////////
// Include files
/////////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <termios.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <time.h>
#include <linux/serial.h>

#include "ocxbpapi.h"

#include "serialport.h"
#include "sputil.h"



/////////////////////////////////////////////////////////////////////////////
// Definitions and Macros
/////////////////////////////////////////////////////////////////////////////
// We cannot open ttyS0, ttyS0 is used for development phase debug purpose
#define SP_NAME_FORMAT "/dev/ttyS%d"

#define STATE_CLOSED 0
#define STATE_OPEN   1



#define USART_CHANNEL_STATUS_REGISTER 			0x0014
#define USART_RECEIVER_HOLDING_REGISTER 		0x0018
#define USART_TRANSMITTER_HOLDING_REGISTER 		0x001C


#define ATMEL_PDC_RCR    0x104
#define ATMEL_PDC_RNCR   0x114

#define ATMEL_PDC_TCR    0x120
#define ATMEL_PDC_TSR    0x124


/////////////////////////////////////////////////////////////////////////////
// Constructed Data Types
/////////////////////////////////////////////////////////////////////////////
// Struct to keep port related data
typedef struct tagPortHandleData
{
    int state;
    int fd;			// File descriptor of the port
    int physicalLayer;
    int memfd;                       // file descriptor for the mem device
    volatile unsigned char *memPage; // pointer to the page memory
    // This queue is used in order to be able to implement the spGets function
    int rxQueueSize;// = 20; //DEFQSIZE;
    char *rxQueue;
    int rxQueueIndex; // = 0;
}PortHandleData;


#define SP_FIRST_PORT      COM1
#define SP_MAX_PORTS       10


// Ports related data. Supports up to SP_MAX_PORTS ports.
PortHandleData portsData[SP_MAX_PORTS];

PFUNC_SP_COMMCALLBACK g_spCommCallback = NULL;



static PortHandleData * spGetPortData(int comport)
{
    if (comport >= SP_FIRST_PORT && comport < SP_MAX_PORTS)
    {
        return &portsData[comport - SP_FIRST_PORT];
    }
    
    return NULL;
}

SPAPIENTRY spRegisterCommCallback(PFUNC_SP_COMMCALLBACK spCallback)
{
    g_spCommCallback = spCallback;
    
    return SP_SUCCESS;
}

SPAPIENTRY spOpen(int comport, BYTE baudrate, BYTE parity, BYTE wordlen, BYTE stopbits, SPAPIHANDLE * spHandle)
{
    char deviceName[32] = "";
    SPAPIHANDLE comHandle = SP_INVALID_HANDLE;
    PortHandleData *portData = spGetPortData(comport);
    
    if (portData == NULL)
    {
        return SP_ERR_NODEVICE;
    }
    
    // Check that port is closed before opening it
    if (portData->state!=STATE_CLOSED)
    {
        return SP_ERR_REOPEN;
    }
    
    // select port device based on port number
    sprintf(deviceName, SP_NAME_FORMAT, comport);
    
    comHandle = comport;
    
    //if (portData->rxQueueSize == 0)
    {
        portData->rxQueueSize = DEFQSIZE;;
    }
    // Allocate memory for the rxQueue
    portData->rxQueue      = malloc(portData->rxQueueSize);
    portData->rxQueueIndex = 0;
    if (portData->rxQueue == NULL) //12/04/2008 - RAR - added memory check
    {
        return SP_ERR_MEMALLOC;
    }
    
    portData->fd = open(deviceName, O_RDWR | O_NOCTTY | O_NDELAY | O_NONBLOCK);
    if (portData->fd == -1 )
    {
        free(portData->rxQueue);  //12/04/2008 - RAR - added memory deallocation
        // If error, return error code
        return SP_ERR_NODEVICE;
    }
    portData->state = STATE_OPEN;
    
    // Open the memory device so we can read and write memory
    spOpenPortMemMap(comport, &portData->memfd, (void **)&portData->memPage);
    
    // Get the port options
    struct termios options;
    tcgetattr(portData->fd,&options);
    // Enable receiver and set local mode
    options.c_cflag |= (CLOCAL | CREAD);
    // Set raw inputs
    options.c_lflag &= ~(ICANON | ECHO | ECHOE | ECHOK | ISIG | ECHONL | IEXTEN);
    // Set new options and flush all buffers
    //tcsetattr(m_iFd,TCSAFLUSH,&options);
    options.c_iflag	= 0;
    options.c_oflag = ~OPOST;
    
    int i;
    for(i=1; i<NCCS; i++)
    {
        options.c_cc[i] = options.c_cc[0];
    }
    options.c_cc[VMIN]  = 1;	//number of chars to wait for
    options.c_cc[VTIME] = 0;	//timeout in deciseconds
    
    int baud;
    switch(baudrate)
    {
    case BAUD_300:
        baud = B300;
        break;
    case BAUD_600:
        baud = B600;
        break;
    case BAUD_1200:
        baud = B1200;
        break;
    case BAUD_2400:
        baud = B2400;
        break;
    case BAUD_4800:
        baud = B4800;
        break;
    case BAUD_9600:
        baud = B9600;
        break;
    case BAUD_19200:
        baud = B19200;
        break;
    case BAUD_38400:
        baud = B38400;
        break;
    case BAUD_57600:
        baud = B57600;
        break;
    case BAUD_115200:
        baud = B115200;
        break;
    case 13:
        baud = B230400;
        break;
    default:
        baud = B19200;
        break;
    }
    
    // Get the databits parameter
    int bits;
    switch(wordlen)
    {
    case WORDLEN5:
        bits = CS5;
        break;
    case WORDLEN6:
        bits = CS6;
        break;
    case WORDLEN7:
        bits = CS7;
        break;
    case WORDLEN8:
        bits = CS8;
        break;
    default:
        bits = CS8;
        break;
    }
    
    // Set the parity
    switch (parity)
    {
    case PARITY_NONE:
        options.c_cflag &= ~PARENB;
        break;
    case PARITY_ODD:
        options.c_cflag |= PARENB;
        options.c_cflag |= PARODD;
        break;
    case PARITY_EVEN:
        options.c_cflag |= PARENB;
        options.c_cflag &= ~PARODD;
        break;
    default:
        options.c_cflag &= ~PARENB;
        break;
    }
    
    switch(stopbits)
    {
    case STOPBITS1:
        options.c_cflag &= ~CSTOPB;
        break;
    case STOPBITS2:
        options.c_cflag |= CSTOPB;
        break;
    default:
        options.c_cflag &= ~CSTOPB;
        break;
    }
    
    // Set the baudrate
    cfsetispeed(&options, baud);
    cfsetospeed(&options, baud);
    
    // Set the data bits
    options.c_cflag &= ~CSIZE;
    options.c_cflag |= bits;
    
    // Set new options and flush all buffers
    tcsetattr(portData->fd,TCSANOW,&options);
    
    int flags;
    flags = fcntl(portData->fd, F_GETFL, 0);
   	flags |= FNDELAY;
   	fcntl(portData->fd, F_SETFL, flags);
    
    // Set the Low Latency flag in the serial port
    struct serial_struct serinfo;
    if (ioctl(portData->fd,TIOCGSERIAL,&serinfo) <0 )
    {
        perror("ioctl TIOCGSERIAL");
    }
    serinfo.flags |= ASYNC_LOW_LATENCY;
    if (ioctl(portData->fd,TIOCSSERIAL,&serinfo) <0)
    {
        perror("ioctl TIOCSSERIAL");
    }
    
    if (spHandle != NULL)
    {
        *spHandle = comHandle;
    }
    
    return SP_SUCCESS;
}


SPAPIENTRY spOpenAlt(int comport, SP_ALT_SETUP *altsetup, SPAPIHANDLE * spHandle)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NODEVICE;
    }
    
    portData->rxQueueSize = altsetup->rxquesize;
    return spOpen(comport, altsetup->baudrate, altsetup->parity, altsetup->wordlen, altsetup->stopbits, spHandle);
}


SPAPIENTRY spClose(SPAPIHANDLE comport)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    close(portData->fd);
    
    spClosePortMemMap(portData->memfd, (void *)portData->memPage);
    
    portData->state = STATE_CLOSED;
    // Deallocate the memory
    free(portData->rxQueue);
    
    return SP_SUCCESS;
}


SPAPIENTRY spSetPortPhysicalLayer(SPAPIHANDLE comport, int physicalLayer)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NODEVICE;
    }
    
    switch (physicalLayer)
    {
    case RS232:
        spSetPortMode(comport, MODE_RS232);
        break;
        
    case RS485:
        spSetPortMode(comport, MODE_RS485R);
        break;
        
    case RS422:
        spSetPortMode(comport, MODE_RS422);
        break;
        
    }
    
    portData->physicalLayer = physicalLayer;
    
    return SP_SUCCESS;
}

SPAPIENTRY spGetPortPhysicalLayer(SPAPIHANDLE comport, int *physicalLayer)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    // Check that pointer is valid
    if (physicalLayer==NULL)
    {
        return SP_ERR_BADPARAM;
    }
    
    *physicalLayer = portData->physicalLayer;
    
    return SP_SUCCESS;
}

// Not supported
// This function allows to set the Automatic RTS option on the ATMEL processor
// state has to be ON or OFF
SPAPIENTRY spSetAutomaticRTS(SPAPIHANDLE comport, int state)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NODEVICE;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Because the polarity of the RTS is inverted for RS485, the
    // automatic RTS can only work for RS485 and not for RS232,
    // So if RS232 is selected, only software controlled RTS is
    // available
    if (portData->physicalLayer == RS232)
    {
        state = OFF;
    }
    
    return SP_SUCCESS;
}


SPAPIENTRY spSetRTS(SPAPIHANDLE comport, int state)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    int status;
    // Get the handshaking signals
    ioctl(portData->fd,TIOCMGET,&status);
    
    if (portData->physicalLayer == RS232)
    {
        switch (state)
        {
        case ON:
            // Set the RTS
            status |= TIOCM_RTS;
            ioctl(portData->fd,TIOCMSET,&status);
            break;
        case OFF:
            // Clear the RTS
            status &= ~TIOCM_RTS;
            ioctl(portData->fd,TIOCMSET,&status);
            break;
        default: return SP_ERR_BADPARAM;
        }
    }
    else if (portData->physicalLayer == RS485)
    {
        switch (state)
        {
        case OFF:
            spSetPortMode(comport, MODE_RS485R);
            break;
        case ON:
            spSetPortMode(comport, MODE_RS485T);
            break;
        default: return SP_ERR_BADPARAM;
        }
    }
    
    if (g_spCommCallback != NULL)
    {
        g_spCommCallback(comport, state == ON ? SP_RTS_ON : SP_RTS_OFF, NULL, 0);
    }
    
    return SP_SUCCESS;
}


SPAPIENTRY spGetRTS(SPAPIHANDLE comport, int *state)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that pointer is valid
    if (state==NULL)
    {
        return SP_ERR_BADPARAM;
    }
    
    int status;
    // Get the handshaking signals
    ioctl(portData->fd,TIOCMGET,&status);
    
    if (portData->physicalLayer == RS232)
    {
        if (status & TIOCM_RTS)
        {
            *state = ON;
        }
        else
        {
            *state = OFF;
        }
    }
    else
    {
        // unsupported for RS422 and RS485
        return SP_ERR_NOACCESS;
        if (status & TIOCM_RTS)
        {
            *state = OFF;
        }
        else
        {
            *state = ON;
        }
    }
    
    return SP_SUCCESS;
}


SPAPIENTRY spSetDTR(SPAPIHANDLE comport, int state)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    int status;
    // Get the handshaking signals
    ioctl(portData->fd,TIOCMGET,&status);
    
    switch (state)
    {
    case ON:
        // Set the DTR
        status |= TIOCM_DTR;
        ioctl(portData->fd,TIOCMSET,&status);
        break;
    case OFF:
        // Clear the DTR
        status &= ~TIOCM_DTR;
        ioctl(portData->fd,TIOCMSET,&status);
        break;
    default: return SP_ERR_BADPARAM;
    }
    
    return SP_SUCCESS;
}


SPAPIENTRY spGetDTR(SPAPIHANDLE comport, int *state)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that pointer is valid
    if (state==NULL)
    {
        return SP_ERR_BADPARAM;
    }
    
    int status;
    // Get the handshaking signals
    ioctl(portData->fd,TIOCMGET,&status);
    
    if (status & TIOCM_DTR)
    {
        *state = ON;
    }
    else
    {
        *state = OFF;
    }
    return SP_SUCCESS;
    
}

SPAPIENTRY spGetCTS(SPAPIHANDLE comport, int *state)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that pointer is valid
    if (state==NULL)
    {
        return SP_ERR_BADPARAM;
    }
    
    int status;
    // Get the handshaking signals
    ioctl(portData->fd,TIOCMGET,&status);
    
    if (status & TIOCM_CTS)
    {
        *state = ON;
    }
    else
    {
        *state = OFF;
    }
    
    if (g_spCommCallback != NULL && *state == ON)
    {
        g_spCommCallback(comport, SP_CTS_ON, NULL, 0);
    }
    
    return SP_SUCCESS;
}

SPAPIENTRY spGetDCD(SPAPIHANDLE comport, int *state)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that pointer is valid
    if (state==NULL)
    {
        return SP_ERR_BADPARAM;
    }
    
    int status;
    // Get the handshaking signals
    ioctl(portData->fd,TIOCMGET,&status);
    
    if (status & TIOCM_CAR)
    {
        *state = ON;
    }
    else
    {
        *state = OFF;
    }
    return SP_SUCCESS;
}


//RAR - 11/24/2008 - Added this new function
// TODO: timeout must be implemnted using select
SPAPIENTRY spPutch(SPAPIHANDLE comport, BYTE ch, DWORD timeout)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    int bytesSent = write(portData->fd, &ch, 1);
    if (bytesSent<0)
    {
        return SP_ERR_NOACCESS;
    }
    else
    {
        if (bytesSent<1)
        {
            return SP_ERR_TIMEOUT;
        }
    }
    
    if (g_spCommCallback != NULL)
    {
        g_spCommCallback(comport, SP_WRITE, &ch, 1);
    }
    
    return SP_SUCCESS;
}


// TODO: timeout must be implemnted using select
SPAPIENTRY spPuts(SPAPIHANDLE comport, BYTE *str, BYTE term, int *len, DWORD timeout)
{
    int rv;
    int bytesSent;
    int slen = 0;
    BYTE * pstr = str;
    
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    if(str == NULL)
    {
        return (SP_ERR_BADPARAM);
    }
    if(len != NULL)
    {
        *len = 0;
    }
    
    rv = SP_SUCCESS;
    while(*str != term)
    {
        bytesSent = write(portData->fd, str, 1);
        if (bytesSent<0)
        {
            return SP_ERR_NOACCESS;
        }
        else
        {
            if (bytesSent<1)
            {
                *len = bytesSent;
                return SP_ERR_TIMEOUT;
            }
            else
            {
                str++;
                slen++;
            }
        }
    }
    
    if(len != NULL)
    {
        *len = slen;
    }
    
    if (g_spCommCallback != NULL)
    {
        g_spCommCallback(comport, SP_WRITE, pstr, slen);
    }
    
    return rv;
}


// TODO: timeout must be implemnted using select
SPAPIENTRY spPutData(SPAPIHANDLE comport, BYTE *data, int *len, DWORD timeout)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    int bytesSent = write(portData->fd, data, *len);
    if (bytesSent<0)
    {
        return SP_ERR_NOACCESS;
    }
    else
    {
        if (bytesSent<*len)
        {
            *len = bytesSent;
            return SP_ERR_TIMEOUT;
        }
    }
    
    if (g_spCommCallback != NULL)
    {
        g_spCommCallback(comport, SP_WRITE, data, *len);
    }
    
    return SP_SUCCESS;
}


//RAR - 11/24/2008 - Added this new function
// TODO: timeout must be implemnted using select
SPAPIENTRY spGetch(SPAPIHANDLE comport, BYTE *ch, DWORD timeout)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    int bytesRead = read(portData->fd, ch, 1);
    if (bytesRead<0)
    {
        return SP_ERR_NOACCESS;
    }
    else
    {
        if (bytesRead<1)
        {
            return SP_ERR_TIMEOUT;
        }
    }
    
    if (g_spCommCallback != NULL)
    {
        g_spCommCallback(comport, SP_READ, ch, 1);
    }
    
    return SP_SUCCESS;
}


// TODO: Fix this routine
SPAPIENTRY spGets(SPAPIHANDLE comport, BYTE *str, BYTE term, int *len, DWORD timeout)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    int ret = SP_ERR_TIMEOUT;
    int rxLen = 1;
    // Count of received characters
    int rxCount = 0;
    
    // Allow to receive up to 10 characters already buffered before
    // returning MVI_ERR_TIMEOUT
    while (rxCount < 10)
    {
        int bytesRead = read(portData->fd, (unsigned char*)&portData->rxQueue[portData->rxQueueIndex], rxLen);
        // If error or no character read return error code
        if (bytesRead < 0)
        {
            return SP_ERR_NOACCESS;
        }
        else
        {
            if (bytesRead<1)
            {
                return SP_ERR_TIMEOUT;
            }
        }
        
        portData->rxQueueIndex++; // One more character was received
        
        // check if we have received the whole string and in that case return it
        // to the user
        if (portData->rxQueue[portData->rxQueueIndex]==term)
        {
            ret = SP_SUCCESS;
            break;
        }
        
        // If client requested length is matched return
        if (portData->rxQueueIndex>=*len)
        {
            ret = SP_SUCCESS;
            break;
        }
        // If queue length is exceeded, return also
        // May require changes in the future
        if (portData->rxQueueIndex >= portData->rxQueueSize)
        {
            ret = SP_SUCCESS;
            break;
        }
        rxCount++;
    }
    
    if (ret == SP_SUCCESS)
    {
        memcpy(str,portData->rxQueue,portData->rxQueueIndex);
        *len = portData->rxQueueIndex;
        portData->rxQueueIndex = 0;
        
        if (g_spCommCallback != NULL)
        {
            g_spCommCallback(comport, SP_READ, str, *len);
        }
    }
    else
    {
        *len = portData->rxQueueIndex;
    }
    
    return ret;
}


// TODO: timeout must be implemnted using select
SPAPIENTRY spGetData(SPAPIHANDLE comport, BYTE *data, int *len, DWORD timeout)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    int bytesRead = read(portData->fd, data, *len);
    if (bytesRead<0)
    {
        return SP_ERR_NOACCESS;
    }
    else
    {
        if (bytesRead<*len)
        {
            *len = bytesRead;
            return SP_ERR_TIMEOUT;
        }
    }
    
    if (g_spCommCallback != NULL)
    {
        g_spCommCallback(comport, SP_READ, data, bytesRead);
    }
    
    return SP_SUCCESS;
}


SPAPIENTRY spGetCountUnsent(SPAPIHANDLE comport, int *count)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that pointer is valid
    if (count==NULL)
    {
        return SP_ERR_BADPARAM;
    }
    
    // get the number of unread input characters
    ioctl(portData->fd, TIOCOUTQ, count);
    
    // A reading of 0 here doesn't mean that all the character have been
    // sent. One extra character can still be at the UART shift register
    // so the UART has to be accessed in order to find if it is done
    if (*count==0)
    {
        int lsr = 0;
        ioctl( portData->fd, TIOCSERGETLSR, &lsr );
        if ((lsr & TIOCSER_TEMT) == 0)
        {
            (*count)++;
        }
    }
    
    return SP_SUCCESS;
}


SPAPIENTRY spGetCountUnread(SPAPIHANDLE comport, int *count)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that pointer is valid
    if (count==NULL)
    {
        return SP_ERR_BADPARAM;
    }
    
    // get the number of unread input characters
    ioctl(portData->fd, FIONREAD, count);
    //printf("In get Count Unread for port: %d, count: %d\n",comport,*count);
    
    return SP_SUCCESS;
}


SPAPIENTRY spSetHandshaking(SPAPIHANDLE comport, int shake)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Get the port options
    struct termios options;
    tcgetattr(portData->fd,&options);
    
    switch(shake)
    {
    case HSHAKE_NONE :
        options.c_cflag &= ~CRTSCTS;
        options.c_iflag &= ~IXON & ~IXOFF;
        break;
    case HSHAKE_RTSCTS :
        options.c_cflag |= CRTSCTS;
        options.c_iflag &= ~IXON & ~IXOFF;
        break;
    case HSHAKE_DTRDSR :
        printf("Software Trying to use DTR/DSR Handshaking. Not implemented\n");
        return SP_ERR_BADPARAM;
        break;
    case HSHAKE_XONXOFF :
        options.c_cflag &= ~CRTSCTS;
        options.c_iflag |= IXON | IXOFF;
        break;
    default: return SP_ERR_BADPARAM;
    }
    // Set new options and flush all buffers
    tcsetattr(portData->fd,TCSANOW,&options);
    
    return SP_SUCCESS;
}


SPAPIENTRY spPurgeDataUnread(SPAPIHANDLE comport)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Flush the Input Buffer
    ioctl(portData->fd, TCFLSH, TCIFLUSH);
    //Flush the ascii buffer
    portData->rxQueueIndex = 0;
    
    return SP_SUCCESS;
}


SPAPIENTRY spPurgeDataUnsent(SPAPIHANDLE comport)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Flush the Output Buffer
    ioctl(portData->fd, TCFLSH, TCOFLUSH);
    
    return SP_SUCCESS;
}


SPAPIENTRY spGetDMACount(SPAPIHANDLE comport, unsigned short int *count)
{
    PortHandleData *portData = spGetPortData(comport);
    if (portData == NULL)
    {
        return SP_ERR_NOACCESS;
    }
    
    // Check that port is open
    if (portData->state!=STATE_OPEN)
    {
        return SP_ERR_NOACCESS;
    }
    
    *count = *(unsigned short *)(portData->memPage + ATMEL_PDC_RCR);
    
    return 0;
}
