/** $Revision:   1.0  $ **/
/**/
/************************************************************************
*
*   Title:  Sample application for the 1756-MVI DH-485 API
*
*   Abstract:
*
*   This sample program implements a simple interface to a Remote DH-485
*   device.  It uses the DH-485 API and transfers data to the device's
*   Common Interface File (CIF).
*
*
*   Environment:    1756-MVI Communications Module
*                   DH-485 API
*                   General Software DOS 6-XL
*                   Borland/Microsoft C/C++ Compiler (16-bit)
*
*   Author:         JJW
*                   Copyright (c) 2000 Online Development, Inc.
*
*
*   Using the program
*   -----------------
*
*   The program allows the serial port, baud rate and node number to
*   be specified on the command line.
*
*   Syntax:
*
*         sample -c[2|3] -b[4|5|7|8] -n[0 to 31]
*
*   Options:
*       Baud rate
*             -b[4]             1200 Baud
*             -b[5]             2400 Baud
*             -b[7]             9600 Baud
*             -b[8]             19200 Baud (Default)
*
*       Communication Port
*             -c2               Indicates that PRT2 is used (Default)
*             -c3               Indicates that PRT3 is used
*
*       Node number = any number in range 0 to 31
*
*
************************************************************************/


/*=======================================================================
=                           INCLUDE FILES                               =
=======================================================================*/

#include    <stdio.h>
#include    <stdlib.h>
#include    <conio.h>
#include    <ctype.h>
#include    <string.h>
#include    "mvispapi.h"
#include    "dh485api.h"


/*=======================================================================
=                      LOCAL SUPPORT PROTOTYPES                         =
=======================================================================*/

void ErrorExit(int);
void SPErrorExit(void);
void ParseCmdLine(int, char **);
int ShowHelp(char *);
int ProcessCommand(void);
void ComPutStr(char *);
void ComGetStr(char *);
int ShowCfg(char *);
int DispANTable(char *);
int DumpCIF(char *);
int WrLocalCIF(char *);
int RdRemoteCIF(char *);
int WrRemoteCIF(char *);

/*=======================================================================
=                    MODULE WIDE GLOBAL VARIABLES                       =
=======================================================================*/

/*** Console Serial Port ***/
int Commport;

MVIHANDLE handle;           // Handle returned from MVIdh_Open function

// port configuration
DH485CONFIG dhcfg = {MVI_COM2, MVI_BAUD_19200, 0, MVI_MODE_MASTER};

/***  Commands  ***/
typedef struct
{
    char    *Cmd;
    int     (*CmdFunct)(char *);
} COMMANDSTRUCT;

COMMANDSTRUCT Commands[] =
{
    { "cfg", ShowCfg },
    { "ant", DispANTable },
    { "rdlcif", DumpCIF },
    { "wrlcif", WrLocalCIF },
    { "rdrcif", RdRemoteCIF },
    { "wrrcif", WrRemoteCIF },
    { "help", ShowHelp },
    { "?", ShowHelp },
    { "quit", NULL },
    { "q", NULL },
    { "", NULL }
};

#define MAXCMDSTRLEN    128     /* maximum length of user-input command string */

/*=======================================================================
=                       LOCAL SUPPORT ROUTINES                          =
=======================================================================*/

/***********************************************************************/
void ComPutStr(char *str)
{
    if (MVIsp_Puts(Commport,(BYTE *)str,0,NULL,1000L) != MVI_SUCCESS)
        SPErrorExit();
}

/***********************************************************************/
void ComGetStr(char *str)
{
    BYTE ch;
    int count;

    count = 0;
    do
    {
        if (MVIsp_Getch(Commport,&ch,TIMEOUT_FOREVER) != MVI_SUCCESS)
            SPErrorExit();
        switch (ch)
        {
            case '\r':      /* cr */
                MVIsp_Putch(Commport,ch,1000L);
                *str++ = '\0';
                break;

            case '\b':      /* backspace */
                if (count > 0)
                {
                    count--;
                    *--str = '\0';
                    MVIsp_Putch(Commport,ch,1000L);
                    MVIsp_Putch(Commport,' ',TIMEOUT_ASAP);
                    MVIsp_Putch(Commport,'\b',TIMEOUT_ASAP);
                }
                break;

            default:
                MVIsp_Putch(Commport,ch,1000L);
                if (count < (MAXCMDSTRLEN - 1))
                {
                    *str++ = ch;
                    count++;
                }
                break;
        }
    }
    while(ch != '\r' );
}

/***********************************************************************/
void ErrorExit(int errcode)
{
    char errbuf[80];

    if (errcode)
    {
        MVIdh_ErrorString(errcode, errbuf);   // get error message
        printf("%s\n", errbuf);
    }

    MVIdh_Close(handle);                   /* should always close before exiting */
    exit(1);
}

/***********************************************************************/
void SPErrorExit(void)
{
    int count;

    /* if serial port error, can't rely on error messages */
    MVIdh_Close(handle);                   /* should always close before exiting */
    do
    {
        MVIsp_GetCountUnsent(Commport,&count);
    }
    while (count);
    MVIsp_Close(Commport);
    exit(1);
}

/********************************************************************
********************************************************************/
void ParseCmdLine(int argc, char *argv[])
{
    int     n;
    char    *arg;
    char    ch;

    for (n=1; n<argc; n++)
    {
        arg = argv[n];

        if ((*arg == '/') || (*arg == '-'))
        {
            ch = *(++arg);
            ch = toupper(ch);
            arg++;
            switch (ch)
            {
                case 'C':                // Test Com port number
                    dhcfg.comport = atoi(arg) - 1;
                    break;

                case 'B':               // Test Baudrate
                    dhcfg.baudrate = atoi(arg);
                    break;

                case 'N':               // Test node number
                    dhcfg.node = atoi(arg);
                    break;

                case '?':
                case 'H':
                    printf("\nUsage: -c[2|3] -b[4|5|7|8] -n[node number]");
                    printf("\nOptions:");
                    printf("\n   -c     Comport for DH-485 communication");
                    printf("\n   -b     specifies baud rate (default=8 or 19200 baud)");
                    printf("\n   -n     specifies the DH-485 node number in range of 0 to 31\n\n");
                    exit(0);
                    break;
            }
        }
    }
}

/******************************************************************************
******************************************************************************/
//#pragma argsused
int ShowHelp(char *tmp)
{
    ComPutStr("\n\r\n\rCommands:");
    ComPutStr("\n\r-------------------------------------------------------------------------------");
    ComPutStr("\n\rcfg               - show the DH-485 Sample Application's configuration");
    ComPutStr("\n\rant               - Dispaly the active node table");
    ComPutStr("\n\rrdlcif            - dump the local CIF data");
    ComPutStr("\n\rwrlcif o d        - write data d into the local CIF at offset o");
    ComPutStr("\n\rrdrcif n o s t    - reads s bytes data d from node n CIF at offset o, waits t");
    ComPutStr("\n\rwrrcif n o d t    - writes data d into node n CIF at offset o, waits t");
    ComPutStr("\n\rhelp,?            - print this Command List");
    ComPutStr("\n\rquit,q            - exit this program\n\r");
    tmp++;      /* suppress compiler warning */
    return(0);
}


/******************************************************************************
******************************************************************************/
int ProcessCommand(void)
{
    static char cCmdBuf[MAXCMDSTRLEN];
    char    tmpbuf[100];
    char    *szArgs;
    char    *szCmd;
    COMMANDSTRUCT *pCommands;
    int CmdSts;

    memset(cCmdBuf,0,100);
    ComGetStr(cCmdBuf);

    /* process command */
    szCmd = strtok(cCmdBuf," \t\r");
    if (strlen(szCmd) == 0)
        return(0);   /* re-prompt if user keeps pressing cr */

    pCommands = (COMMANDSTRUCT *)&Commands;
    while (strlen(pCommands->Cmd) > 0)
    {
        if (stricmp(pCommands->Cmd,szCmd) == 0)
        {
            if (pCommands->CmdFunct != NULL)
            {
                szArgs = &(cCmdBuf[strlen(szCmd)+1]);
                CmdSts = (pCommands->CmdFunct)(szArgs);
                if (CmdSts == 1)
                    ComPutStr("\n\rError #1 -- Bad Syntax");
                else if (CmdSts > 1)
                {
                    sprintf(tmpbuf,"\n\rError %d",CmdSts);
                    ComPutStr(tmpbuf);
                }
            }
            else if (szCmd[0] == 'q' || szCmd[0] == 'Q')
                return(1);  /* exit the program */
            return(0);
        }
        pCommands++;
    }
    /* command not found */
    ComPutStr("\n\rHuh? ('?' gives Command List)");
    return(0);
}

/*=======================================================================
=                       MAIN ENTRY POINT                                =
=======================================================================*/

/************************************************************************
*
*     Entry point:
*       main
*
*     Description:
*
*     Arguments:
*       none
*
*     External effects:
*
*
*     Return value:
*       none
*
*-----------------------------------------------------------------------
*     Notes:
*
*     debugging/error printf's will only be seen if console is enabled.
*
************************************************************************/
void main(int argc, char *argv[])
{
    int  CmdSts;
    int rc;
    DH485VERSIONINFO dh485ver;

    // Set console serial port to Com1
    Commport = COM1;

    ParseCmdLine(argc, argv);

    /* Open a serial port console */
    rc = MVIsp_Open(Commport,BAUD_19200,PARITY_NONE,WORDLEN8,STOPBITS1);
    if (rc != MVI_SUCCESS)
    {
        printf("\n\rError Opening Console Serial Port %d\n\r",rc);
        exit(1);
    }

    // Set the DH-485 mode to Master
    dhcfg.mode = MVI_MODE_MASTER;

    // Open the DH-485 API
    if (MVI_SUCCESS != (rc = MVIdh_Open(&handle, &dhcfg)))
    {
        printf("\n\rMVIdh_Open failed: %d\n\r", rc);
        ErrorExit(rc);
    }

    /* print a sign-on message */
    printf("\n\r1756-MVI DH-485 API Sample Program V1.0");

    /*
    ** Display the DH-485 API version
    */
    if (MVI_SUCCESS != (rc = MVIdh_GetVersionInfo(handle, &dh485ver)))
    {
        printf("\n\rMVIdh_GetVersionInfo failed: %d\n\r", rc);
        ErrorExit(rc);
    }
    printf("\n\r\n\rDH-485 API Version %d.%d", dh485ver.APISeries, dh485ver.APIRevision);

    printf("\n\n\r--Enter 'help' or '?' to show menu\n\r");

    /*
    ** this app is now online, process user's commands
    */
    do
    {
        /* display prompt */
        ComPutStr("\n\rdh845> ");

        /* get a command from user */
        CmdSts = ProcessCommand();

    } while (CmdSts == 0);

    /* Close the API
    ** No need to print any error messages here since we
    ** are closing the API
    */
    MVIdh_Close(handle);

}

/********************************************************************
********************************************************************/

int ShowCfg(char *szTmp)
{
    int rc;
    DH485VERSIONINFO dhver;
    MVISPVERSIONINFO spver;
    BYTE bStatus;
    char tmpbuf[100];

    szTmp++;        /* suppress compiler warning */

    /*
    ** Display the Serial Port API version
    */
    if (MVI_SUCCESS != (rc = MVIsp_GetVersionInfo(&spver)))
    {
        sprintf(tmpbuf,"\n\rMVIsp_GetVersionInfo failed: %d\n\r", rc);
        ComPutStr(tmpbuf);
        return(rc);
    }
    sprintf(tmpbuf,"\n\r\n\rSerial Port API Version %d.%d", spver.APISeries, spver.APIRevision);
    ComPutStr(tmpbuf);

    /*
    ** Display the DH-485 API version
    */
    if (MVI_SUCCESS != (rc = MVIdh_GetVersionInfo(handle, &dhver)))
    {
        sprintf(tmpbuf,"\n\rMVIdh_GetVersionInfo failed: %d\n\r", rc);
        ComPutStr(tmpbuf);
        return(rc);
    }
    sprintf(tmpbuf,"\n\rDH485 API Version %d.%d",dhver.APISeries, dhver.APIRevision);
    ComPutStr(tmpbuf);

    // get communication status
    if ( MVI_SUCCESS != (rc = MVIdh_GetCommStatus(handle, &bStatus)) )
    {
        sprintf(tmpbuf,"\n\rMVIdh_GetCommStatus failed: %d\n\r", rc);
        ComPutStr(tmpbuf);
        return(rc);
    }

    if( bStatus == MVI_COMM_STATUS_ON )
        sprintf(tmpbuf,"Communication status:      ONLINE");
    else
        sprintf(tmpbuf,"Communication status:      OFFLINE");

    /* Display Connection Status */
    ComPutStr("\n\r\n\r");
    ComPutStr(tmpbuf);

    ComPutStr("\n\r");
    return(0);
}

/********************************************************************
********************************************************************/
int DispANTable(char *szTmp)
{
    int i, rc;
    DWORD ANTable;
    char tmpbuf[100];

    szTmp++;  /* suppress warning */

     // Retrieve the active node table
    if (MVI_SUCCESS != (rc = MVIdh_GetANTable(handle, &ANTable)))
    {
        sprintf(tmpbuf,"\nMVIdh_GetANTable failed: %d\n", rc);
        ComPutStr(tmpbuf);
        return(rc);
    }

    ComPutStr("\n\r\n\r");

    // Dispaly 1st half of the active node table
    sprintf(tmpbuf, " [00..15] ");
    ComPutStr(tmpbuf);
    for( i=0; i<16; i++ )
    {
        if( (ANTable & (0x00000001L<<i)) == 0 )
            sprintf(tmpbuf, "  .");
        else
            sprintf(tmpbuf, " %02d", i);

        ComPutStr(tmpbuf);
    }

    // Dispaly 2nd half of the active node table
    sprintf(tmpbuf, "\n\r [16..31] ");
    ComPutStr(tmpbuf);
    for( i=16; i<32; i++ )
    {
        if( (ANTable & (0x00000001L<<i)) == 0 )
            sprintf(tmpbuf, "  .");
        else
            sprintf(tmpbuf, " %02d", i);

        ComPutStr(tmpbuf);
    }

    ComPutStr("\n\r");

    return(0);
}

/********************************************************************
********************************************************************/
int DumpCIF(char *szTmp)
{
    int  rc;
    WORD lineSize;
    WORD offset = 0;
    WORD dataSize=244;  // maximum size of CIF
    BYTE dataBuf[244];
    char tmpbuf[100];

    szTmp++;  /* suppress warning */


    // Retrieve data from the CIF file
    if (MVI_SUCCESS != (rc = MVIdh_GetDataFromCIF(handle, offset, dataSize, &dataBuf[0])))
    {
        sprintf(tmpbuf,"\n\rMVIdh_GetDataFromCIF failed: %d\n", rc);
        ComPutStr(tmpbuf);
        return(rc);
    }

    ComPutStr("\n\r\n\r");

    // Successful read of data, display it
    while( dataSize )
    {
        lineSize = 16;

        sprintf( tmpbuf, "[%02X]  ", offset );
        ComPutStr(tmpbuf);

        while( lineSize && dataSize )
        {
            sprintf( tmpbuf, "%02X ", dataBuf[offset]);
            ComPutStr(tmpbuf);
            lineSize--;
            dataSize--;
            offset++;
        }
        sprintf(tmpbuf,"\n\r");
        ComPutStr(tmpbuf);
    }
    return(0);
}

/********************************************************************
********************************************************************/
int WrLocalCIF(char *szTmp)
{
    int  rc;
    char *p;
    WORD offset;
    WORD dataSize=1;
    BYTE data;
    char tmpbuf[100];

    /* parse cmdline for offset */
    p = strtok(szTmp," \t\r,");
    if (strlen(p) == 0)
        return(1);          /* syntax error */
    offset = atoi(p);

    /* parse cmdline for data */
    p = strtok(NULL," \t\r,");
    if (strlen(p) == 0)
        return(1);          /* syntax error */
    data = (BYTE)atoi(p);

    // Put into the CIF file
    if (MVI_SUCCESS != (rc = MVIdh_PutDataToCIF(handle, &data, offset, dataSize )))
    {
        sprintf(tmpbuf,"\n\rMVIdh_PutDataToCIF failed: %d\n", rc);
        ComPutStr(tmpbuf);
        return(rc);
    }

    ComPutStr("\n\r\n\r");
    sprintf(tmpbuf,"Success\n\r");
    ComPutStr(tmpbuf);

    return(0);
}

/********************************************************************
********************************************************************/
int RdRemoteCIF(char *szTmp)
{
    int  rc;
    char *p;
    WORD lineSize;
    WORD node;
    WORD offset;
    WORD size;
    WORD timeout;
    BYTE pData[244];
    char tmpbuf[100];

    /* parse cmdline for node */
    p = strtok(szTmp," \t\r,");
    if (strlen(p) == 0)
        return(1);          /* syntax error */
    node = atoi(p);
    if (node > 31 )
        return(1);          /* syntax error */

    /* parse cmdline for offset */
    p = strtok(NULL," \t\r,");
    if (strlen(p) == 0)
        return(1);          /* syntax error */
    offset = atoi(p);

    /* parse cmdline for data size */
    p = strtok(NULL," \t\r,");
    if (strlen(p) == 0)
        return(1);          /* syntax error */
    size = atoi(p);

    /* parse cmdline for timeout */
    p = strtok(NULL," \t\r,");
    if (strlen(p) == 0)
        return(1);          /* syntax error */
    timeout = atoi(p);

    // Send the request
    if (MVI_SUCCESS != (rc = MVIdh_ReadRemoteCIFFile( handle, node, offset, size, &pData[0], timeout )))
    {
        sprintf(tmpbuf,"\n\rMVIdh_ReadRemoteCIFFile failed: %d\n", rc);
        ComPutStr(tmpbuf);
        return(rc);
    }

    ComPutStr("\n\r\n\r");

    // Successful read of data, display it
    while( size )
    {
        lineSize = 16;

        sprintf( tmpbuf, "[%02X]  ", offset );
        ComPutStr(tmpbuf);

        while( lineSize && size )
        {
            sprintf( tmpbuf, "%02X ", pData[offset]);
            ComPutStr(tmpbuf);
            lineSize--;
            size--;
            offset++;
        }
        sprintf(tmpbuf,"\n\r");
        ComPutStr(tmpbuf);
    }

    return(0);
}


/********************************************************************
********************************************************************/
int WrRemoteCIF(char *szTmp)
{
    int  rc;
    char *p;
    WORD node;
    WORD offset;
    WORD timeout;
    BYTE Data;
    char tmpbuf[100];

    /* parse cmdline for node */
    p = strtok(szTmp," \t\r,");
    if (strlen(p) == 0)
        return(1);          /* syntax error */
    node = atoi(p);

    /* parse cmdline for offset */
    p = strtok(NULL," \t\r,");
    if (strlen(p) == 0)
        return(1);          /* syntax error */
    offset = atoi(p);

    /* parse cmdline for data */
    p = strtok(NULL," \t\r,");
    if (strlen(p) == 0)
        return(1);          /* syntax error */
    Data = atoi(p);

    /* parse cmdline for timeout */
    p = strtok(NULL," \t\r,");
    if (strlen(p) == 0)
        return(1);          /* syntax error */
    timeout = atoi(p);

    // Send the request
    if (MVI_SUCCESS != (rc = MVIdh_WriteRemoteCIFFile( handle, &Data, node, offset, sizeof(BYTE), timeout )))
    {
        sprintf(tmpbuf,"\n\MVIdh_WriteRemoteCIFFile failed: %d\n", rc);
        ComPutStr(tmpbuf);
        return(rc);
    }

    ComPutStr("\n\r\n\r");
    sprintf(tmpbuf,"Success\n\r");
    ComPutStr(tmpbuf);

    return(0);
}

