	TITLE	IOMULTI - Multi-threaded File I/O Stress Test.

;***	IOMULTI - Multi-threaded File I/O Stress Test.
;
;1.	Functional Description.
;	This program spawns three threads and has them simultaneously
;	create a new file of some random length between 1 and 1MB, copy
;	it to a new file, compare the two files, and then delete them.
;	The process repeats itself indefinitely.  Errors are flagged in
;	each thread's separate error log, which it creates in directory
;	".\ERROR".
;
;2.	Modification History.
;	S. E. Jones	91/01/27.	For stress-testing multisector I/O.
;
;3.	NOTICE: Copyright (C) 1991 General Software.
;
;4.	Build Environment.
;	MASM 5.10, no special switches.

	include ..\inc\usegs.inc
	include ..\inc\udefines.inc
	include ..\inc\umacros.inc
	include ..\inc\ustruc.inc
	include ..\inc\dosapi.inc
	include ..\inc\doserr.inc
	include ..\inc\kernel.inc

THREAD_COUNT =	2			; # threads running in test.

ERRORMSG MACRO	msg
	LOCAL	addr
BLAH	SEGMENT PARA PUBLIC 'DATA'
addr	db	msg			; the message itself.
	db	0			; zero-byte terminator.
BLAH	ENDS
	mov	MsgAddr.[bp], OFFSET BLAH:addr
	Pcall	DisplayMsg		; displays our message.
	ENDM

;	Define the stack.

_STACK	SEGMENT PARA STACK 'STACK'
	db	1024 dup ('$')
TopStack =	$
_STACK	ENDS

UCODE	SEGMENT

ThreadId	dw	1		; threads pick this up & increment it.
DestroyCount	dw	0		; nonzero means threads should destruct.
FailFlag	dw	0		; set to 1 if failure.

;	The following defines local variables on each thread's stack frame,
;	referenced by (BP).

BUFSIZE         =	256		; maximum buffer size in bytes.

AUTO		struc
MsgAddr         dw	?		; ofs FWA, error msg.
FileSize	dw	?		; # bytes in file's size.
Pass		dw	?		; # test iterations.
Datum		db	?		; datum to fill in buffer.
		db	?		; for alignment.
InHandle	dw	?		; file handle for input file.
OutHandle	dw	?		; file handle for output file.
File1		db	16 dup (?)	; filename manufactured by 5Ah.
File2		db	16 dup (?)	; filename manufactured by 5Ah.
Buffer		db	BUFSIZE dup (?) ; buffer for I/O.
AUTO		ends

;***	DisplayMsg - Display Error Message.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine is called by some thread to display a status message.
;
;   MODIFICATION HISTORY.
;	S. E. Jones	91/01/27.	For stress-testing multisector I/O.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	MsgAddr - FWA, msg to be displayed.
;
;   EXIT.
;	none.
;
;   USES.
;	all.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc DisplayMsg
	push	ax
	push	es
	mov	dl, SYS_ENTER_CRITICAL_SECTION
	int	SYSINT			; disallow preemption.

	mov	ax, BLAH
	mov	es, ax			; (ES) = seg FWA, BLAH segment.

	mov	al, Datum.[bp]
	sub	ah, ah			; (AX) = our thread ID.

	PRINTF	<Thread $u, Pass $u: $s\n>, <ax, Pass.[bp], es, MsgAddr.[bp]>

	mov	dl, SYS_LEAVE_CRITICAL_SECTION
	int	SYSINT			; let other threads in.
	pop	es
	pop	ax
EndProc DisplayMsg

;***	CreateNewFile - Create New File and Open It.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine is called by the thread body routine to create a new
;	thread using DOS function 5Ah and fill it with our special byte.
;	The opened file's handle is returned in InHandle.
;
;   MODIFICATION HISTORY.
;	S. E. Jones	91/01/27.	For stress-testing multisector I/O.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	none.
;
;   EXIT.
;	CY	- set if failure, else clear if success.
;	AX	- if failure, DOSERR status code.
;	InHandle - set to open file handle if success.
;
;   USES.
;	all.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc CreateNewFile
	ERRORMSG 'Creating input file.'
	mov	byte ptr File1.[bp+0], '.'
	mov	byte ptr File1.[bp+1], '\'
	mov	word ptr File1.[bp+2], 0
	mov	word ptr File1.[bp+4], 0
	mov	word ptr File1.[bp+6], 0
	mov	word ptr File1.[bp+8], 0
	mov	word ptr File1.[bp+10], 0
	mov	word ptr File1.[bp+12], 0
	mov	word ptr File1.[bp+14], 0
	mov	word ptr File1.[bp+16], 0

	mov	ah, DOSCREATETEMPFILE
	sub	cx, cx			; (CX) = temp file attribute.
	lea	dx, File1.[bp]		; (DS:DX) = FWA, ASCIIZ filename prefix.
	int	21h			; create the new file.
	jc	CreateNewFile_CreateFail; if failure.
	mov	InHandle.[bp], ax	; save handle for the future.

;	Now write "FileSize" bytes to the file.

	mov	cx, FileSize.[bp]	; (CX) = # bytes to write.
	lea	dx, Datum.[bp]		; (DS:DX) = FWA, buffer to write from.
	mov	bx, InHandle.[bp]	; (BX) = handle to write to.
@@:	mov	si, cx			; (SI) = saved bytes to write.
	mov	ah, DOSWRITE		; (AH) = WRITE FILE function code.
	mov	cx, 1			; (CX) = 1 byte/this write.
	int	21h			; write the byte to the file.
	jc	CreateNewFile_WriteFail ; if failure.
	mov	cx, si			; (CX) = restored bytes left to go.
	loop	@b			; do the rest.
	ret				; return successfully.

CreateNewFile_CreateFail:
	ERRORMSG '(CreateNewFile) Unable to create new input file.'
	mov	FailFlag, 1
	ret

CreateNewFile_WriteFail:
	ERRORMSG '(CreateNewFile) Unable to write new input file.'
	mov	FailFlag, 1
EndProc CreateNewFile

;***	CopyFile - Copy New File To Another New File.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine is called by the thread body routine to copy a newly
;	created file to another file.
;
;   MODIFICATION HISTORY.
;	S. E. Jones	91/01/27.	For stress-testing multisector I/O.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	InHandle - set to open file handle if success.
;
;   EXIT.
;	CY	- set if failure, else clear if success.
;	AX	- if failure, DOSERR status code.
;
;   USES.
;	all.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc CopyFile
	ERRORMSG 'Copying file.'
	mov	byte ptr File2.[bp+0], '.'
	mov	byte ptr File2.[bp+1], '\'
	mov	word ptr File2.[bp+2], 0
	mov	word ptr File2.[bp+4], 0
	mov	word ptr File2.[bp+6], 0
	mov	word ptr File2.[bp+8], 0
	mov	word ptr File2.[bp+10], 0
	mov	word ptr File2.[bp+12], 0
	mov	word ptr File2.[bp+14], 0
	mov	word ptr File2.[bp+16], 0
	mov	ah, DOSCREATETEMPFILE
	sub	cx, cx			; (CX) = temp file attribute.
	lea	dx, File2.[bp]		; (DS:DX) = FWA, ASCIIZ filename prefix.
	int	21h			; create the new file.
	LJC	CopyFile_CreateFail	; if failure.
	mov	OutHandle.[bp], ax	; save handle for the future.

;	Reset the input file's file pointer.

	mov	ah, DOSSETPOS		; (AH) = seek file function code.
	mov	al, SEEK_MODE_BOF	; (AL) = mode, from beginning of file.
	mov	bx, InHandle.[bp]	; (BX) = file handle to seek on.
	sub	cx, cx
	sub	dx, dx			; (CX:DX) = (0:0) file pointer to set.
	int	21h			; set the file pointer.
	LJC	CopyFile_SeekFail

;	Now copy the file.

	mov	si, FileSize.[bp]	; (SI) = down-counter of bytes to copy.
CopyFile_Loop:
	or	si, si			; are we done yet?
	jz	CopyFile_Exit		; if so.
	mov	cx, si			; (CX) = total bytes left/this file.
	mov	ax, BUFSIZE		; (AX) = maximum buffer size in bytes.
	sub	cx, ax
	sbb	dx, dx
	and	cx, dx
	add	cx, ax			; (CX) = actual bytes we can do at once.

;	Read one buffer's full.

	mov	ah, DOSREAD
	mov	bx, InHandle.[bp]	; (BX) = handle to read from.
	lea	dx, Buffer.[bp]         ; (DS:DX) = FWA, buffer to read into.
	int	21h			; do the read.
	LJC	CopyFile_ReadFail	; if failure.

;	Write one buffer's full.

	mov	cx, ax			; (CX) = bytes read from file.
	mov	ah, DOSWRITE
	mov	bx, OutHandle.[bp]	; (BX) = handle to write to.
	int	21h			; do the write.
	LJC	CopyFile_WriteFail	; if failure.
	sub	si, ax			; (SI) = bytes left to copy.
	jmp	CopyFile_Loop		; go do the rest.

CopyFile_Exit:
	mov	ah, DOSCLOSE
	mov	bx, InHandle.[bp]
	int	21h			; close input file handle.
	jnc	@f			; if success.
	ERRORMSG '(CopyFile) Unable to close input file.'
	mov	FailFlag, 1

@@:	mov	ah, DOSDELETE
	lea	dx, File1.[bp]
	int	21h			; delete the input file.
	jnc	@f			; if success.
	ERRORMSG '(CopyFile) Unable to delete input file.'
	mov	FailFlag, 1

@@:	mov	ah, DOSCLOSE
	mov	bx, OutHandle.[bp]
	int	21h			; close output file handle.
	jnc	@f			; if success.
	ERRORMSG '(CopyFile) Unable to close output file.'
	mov	FailFlag, 1

@@:	mov	ah, DOSDELETE
	lea	dx, File2.[bp]
	int	21h			; delete the input file.
	jnc	@f			; if success.
	ERRORMSG '(CopyFile) Unable to delete output file.'
	mov	FailFlag, 1
@@:	ret				; return to caller.

CopyFile_CreateFail:
	ERRORMSG '(CopyFile) Unable to create output file.'
	mov	FailFlag, 1
	jmp	CopyFile_Exit

CopyFile_SeekFail:
	ERRORMSG '(CopyFile) Unable to seek on input file.'
	mov	FailFlag, 1
	jmp	CopyFile_Exit

CopyFile_ReadFail:
	ERRORMSG '(CopyFile) Unable to read input file.'
	mov	FailFlag, 1
	jmp	CopyFile_Exit

CopyFile_WriteFail:
	ERRORMSG '(CopyFile) Unable to write output file.'
	mov	FailFlag, 1
	jmp	CopyFile_Exit
EndProc CopyFile

;***	Thread - Thread Body.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine is executed reentrantly by all the threads.
;
;   MODIFICATION HISTORY.
;	S. E. Jones	91/01/27.	For stress-testing multisector I/O.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	none.
;
;   EXIT.
;	none.
;
;   USES.
;	all.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc Thread
	sub	sp, (SIZE AUTO)         ; give ourself a big stack.
	mov	bp, sp			; (BP) = our stack frame.

	push	ss
	pop	ds			; (DS) = stack frame.

	cli
	mov	ax, ThreadId
	inc	ThreadId
	mov	Datum.[bp], al		; save our storage value.
	sti

	mov	Pass.[bp], 1		; reset pass number.
	mov	FileSize.[bp], 1	; maximum file size.
	ERRORMSG 'Thread started.'

Thread_Loop:
	cmp	DestroyCount, 0         ; should we destroy ourselves?
	jne	Thread_Destruct         ; if so.

	mov	InHandle.[bp], 99	; in case we fail.
	mov	OutHandle.[bp], 99	; in case we fail.

	Pcall	CreateNewFile
	Pcall	CopyFile

	cmp	FailFlag, 0		; did anyone fail?
	jne	Thread_Destruct         ; if so.

	ERRORMSG 'Successful pass.'
	inc	Pass.[bp]		; count the passes.
	inc	FileSize.[bp]		; keep trying bigger files.
	jmp	Thread_Loop		; go do another iteration.

Thread_Destruct:
	dec	DestroyCount		; indicate we're gone.
	PRINTF	<Destroying thread.\n>
	DeallocateThread		; self-destruct.
EndProc Thread

;***	Main - Main Entrypoint.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine is the entrypoint of the iomulti stress program.
;
;   MODIFICATION HISTORY.
;	S. E. Jones	91/01/27.	For stress-testing multisector I/O.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	none.
;
;   EXIT.
;	none.
;
;   USES.
;	all.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc Main, PUBLIC, FAR
	mov	ax, DGROUP
	mov	ds, ax
	ASSUME	DS:DGROUP		; (DS) = DGROUP.

	PRINTF	<IOMULTI Stress Test.  Three threads will be spawned now.\n>
	PRINTF	<To terminate, hit any key (threads will evaporate soon).\n\n>

	mov	DestroyCount, 0         ; threads shouldn't destroy themselves.

	mov	cx, THREAD_COUNT	; (CX) = # threads to allocate.
Main_Loop:
	AllocateThread Thread
	loop	Main_Loop

;	Spin trying to read a console character, or until FailFlag=TRUE.

@@:	cmp	FailFlag, 0		; did someone fail?
	jne	Main_Exit		; if so.
	mov	ah, DOSCONIO		; wait for operator.
	mov	dl, 0ffh		; (DL) = get char function code.
	int	21h			; clears ZF if char from keyboard.
	jz	@b			; if no chars yet.

	mov	DestroyCount, THREAD_COUNT ; down-counter of threads to destroy.
	sti
@@:	cmp	DestroyCount, 0         ; have they all destroyed themselves?
	jne	@b			; if not.

Main_Exit:
	PRINTF	<IOMULTI: Terminated.\n>

;	Now terminate the process and return to STARLITE.

	mov	ah, DOSEXIT
	mov	al, 0			; successful status code.
	int	21h			; terminate program.
EndProc Main

UCODE	ENDS
	END	Main
